/***********************************************************************
    Authors: Sparks (Gameplay Scripting) + AmishThunder (Level Design)
    Map: Subway
    Campaign: USMC
    Objectives:
    	1) Defuse the bombs in time
***********************************************************************/

#include common_scripts\utility;
#include maps\_utility;
#include maps\_anim;

main()
{
	/*--------------------
	 CAMPAIGN
	----------------------*/
	level.campaign = "american";
	
	/*--------------------
	 START POINTS
	----------------------*/
	// Handled by main so declare before _load
	add_start( "intro", ::start_intro, &"SUBWAY_START_INTRO" );
	add_start( "mission", ::start_mission, &"SUBWAY_START_MISSION" );
	add_start( "boss", ::start_boss, &"SUBWAY_START_BOSS" );
	add_start( "explosion", ::start_explosion, &"SUBWAY_START_EXPLOSION" );
	add_start( "credits", ::start_credits, &"SUBWAY_START_CREDITS" );
	default_start( ::start_intro );
	
	/*--------------------
	 GLOGBAL SCRIPTS
	----------------------*/
	maps\_load::main();
	maps\_compass::setupMiniMap( "compass_map_mp_subway2" );
	maps\_nightvision::main();
	maps\subway_anim::main();
	maps\subway_amb::main();
	maps\subway_fx::main();
	maps\_drone::init();
	animscripts\dog_init::initDogAnimations();
	
	level.weaponClipModels = [];
	level.weaponClipModels[0] = "weapon_ak47_clip";
	level.weaponClipModels[1] = "weapon_ak74u_clip";
	level.weaponClipModels[2] = "weapon_m16_clip";
	level.weaponClipModels[3] = "weapon_saw_clip";
	level.weaponClipModels[4] = "weapon_rpd_clip";
	level.weaponClipModels[5] = "weapon_g3_clip";
	
	/*--------------------
	 PRECACHE
	----------------------*/
	precacheString(&"SUBWAY_INTROSCREEN_LINE_1");
	precacheString(&"SUBWAY_INTROSCREEN_LINE_2");
	precacheString(&"SUBWAY_INTROSCREEN_LINE_3");
	precacheString(&"SUBWAY_INTROSCREEN_LINE_4");
	precacheString(&"SUBWAY_PRESS_DEFUSE");
	precacheString(&"SUBWAY_OBJECTIVE");
	precacheString(&"SUBWAY_TIME_TO_DEFUSE_BOMBS");
	precacheString(&"SUBWAY_CREDIT1");
	precacheString(&"SUBWAY_CREDIT2");
	precacheString(&"SUBWAY_CREDIT3");
	precacheString(&"SUBWAY_START_INTRO");
	precacheString(&"SUBWAY_START_MISSION");
	precacheString(&"SUBWAY_START_BOSS");
	precacheString(&"SUBWAY_START_CREDITS");
	precacheshellshock( "aftermath" );
	precacheshader("white");
	precacheshader("black");
	
	/*--------------------
	 FLAGS
	----------------------*/
	//Intro
	flag_init( "introscreen_done" );
	flag_init( "failed" );
	flag_init( "boss_sequence_is_last" );
	
	/*--------------------
	 MAIN CHARACTERS
	----------------------*/
	level.vaz = getEnt ( "vaz", "targetname" );
	level.vaz.animName = "vaz";
	level.vaz thread magic_bullet_shield();
	level.griggs = getEnt ( "griggs", "targetname" );
	level.griggs.animName = "griggs";
		
	/*--------------------
	 COLOR GROUPS
	----------------------*/
	// When orange guys die, red guys replace orange guys
	// Little over a week and half to do the map, so this may not be fully implemented in the map but it's here
	clear_promotion_order();
	set_promotion_order( "o", "r" );
	red_guys = get_force_color_guys( "allies", "r" );
	array_thread( red_guys, ::replace_on_death );
	
	/*--------------------
	 HINTS
	----------------------*/
	add_hint_string( "nvg", &"SUBWAY_NIGHTVISION_USE", maps\_nightvision::ShouldBreakNVGHintPrint );
	
	/*--------------------
	 GLOBAL THREADS
	----------------------*/
	// Stuff that always need to be called, regardless of start points
	thread player_loadout(); // Give the player some weapons
	thread train_rumble(); // Emulates other trains moving nearby in tunnels
	
	/*--------------------
	 MISC
	----------------------*/
	// Civilians
	civilians = getEntArray( "civ_models", "targetname" );
	assertEx( isDefined( civilians ), "You are missing civilians." );
	array_thread( civilians, ::civilian_think );
	
	// Friendlies
	ai = getaiarray("allies");
	for( x = 0 ; x < ai.size ; x++ )
		ai[ x ].health = 1001;
	
	// Fan Explosion Setup
	level.fan_a = getEnt( "turbofan_after", "targetname" );
	level.fan_b = getEnt( "turbofan_before", "targetname" );
	assertEx( isDefined( level.fan_a ), "Missing fan after." );
	assertEx( isDefined( level.fan_b ), "Missing fan before." );
	level.fan_a hide();
	level.fan_frame_a = getEnt( "frame_after", "targetname" );
	level.fan_frame_b = getEnt( "frame_before", "targetname" );
	assertEx( isDefined( level.fan_frame_a ), "Missing fan frame after." );
	assertEx( isDefined( level.fan_frame_b ), "Missing fan frame before." );
	level.fan_frame_a hide();
} // main()

/***********************************************************************
    START FUNCTIONS
***********************************************************************/ 

start_intro()
{
	// Opening sequence for map is setup and threaded from here
	level.player setorigin( level.player.origin );
	level.player setplayerangles( level.player.angles );
	level.player freezeControls( true );
	level.player allowsprint( false );
	
	thread introscreen();
	thread fan_explosion();
} // start_intro()

start_mission()
{
	// Right after opening sequence, quickly start the mission
	level.player setorigin( level.player.origin );
	level.player setplayerangles( (0, 270, 0) );
	
	thread lev_timer();
} // start_mission()

start_boss()
{
	// Ventilation explosion sequence
	level.player setorigin( (3584, -2864, 16) );
	level.player setplayerangles( (0, 270, 0) );
	
	thread fan_explosion();
} // start_boss()

start_explosion()
{
	// Test explosion if player runs out of time
	level.player setorigin( level.player.origin );
	level.player setplayerangles( level.player.angles );
	
	// 3 seconds before kaboom!
	thread timer_director( 3, &"SUBWAY_TIME_TO_DEFUSE_BOMBS" );
} // start_mission()

start_credits()
{
	// Test the credits
	level.player setorigin( level.player.origin );
	level.player setplayerangles( level.player.angles );
	
	flag_set( "boss_sequence_is_last" );	
	thread credits();
} // start_credits()

/***********************************************************************
    LEVEL FUNCTIONS
***********************************************************************/ 

ini_dialogue()
{
	/*--------------------
	 LT. VASQUEZ GIVING OPENING SPEECH
	----------------------*/
	flag_wait( "introscreen_done" );
	
	player_lock = spawn( "script_model", level.player.origin );
	level.player playerlinktodelta ( player_lock );
	player_lock hide();
	
	level.vaz anim_single_solo( level.vaz, "rally" );
	wait 0.2;
	level.griggs anim_single_solo( level.griggs, "watchsix" );
	
	level.player unlink();
	player_lock delete();
	
	wait 1;
	level.player allowsprint( true );
	thread lev_timer();
} // ini_dialogue()

#using_animtree("generic_human");
civilian_think()
{
	/*--------------------
	 SET UP CIVILIAN ANIMS
	----------------------*/
	assertEx( isDefined( self.script_noteworthy ), "Missing animation value at: " + self.origin );
	civAnim = self.script_noteworthy;
	assertEx( isDefined( level.scr_anim[ "generic" ][ civAnim ] ), "You are missing or do not have animation: " + civAnim );
	self.animName = "generic";
	self UseAnimTree(#animtree);
	
	if( isDefined( self.target ) )
	{
		node = getnode( self.target, "targetname" );
		assertEx( isDefined( node ), "Missing or corrupted node at: " + self.origin );
		node anim_generic_loop( self, civAnim );
	}
	else
		thread anim_generic_loop( self, civAnim );
} // civilian_think()

objectives()
{
	/*--------------------
	 GET OBJECTIVE TRIGGERS
	----------------------*/
	level.obj_loco = undefined;
	level.obj_loco = getEntarray( "bomb_trig", "targetname" );
	assert( isDefined( level.obj_loco ) );
		
	for( x = 0 ; x < level.obj_loco.size ; x++)
	{
		level.obj_loco[ x ].completed = false;
		level.obj_loco[ x ] SetHintString( &"SUBWAY_PRESS_DEFUSE" );
	}
	
	/*--------------------
	 INITIALIZE OBJECTIVES 
	----------------------*/
	thread update_objective_points();
	thread bomb_triggers();
	MusicStop();
	MusicPlayWrapper("mission_song");
	thread display_hint( "nvg" );
} // objectives()

update_objective_points()
{
	/*--------------------
	 UPDATE OBJECTIVE MAP POINTS
	----------------------*/
	obj_string = 0;
	objective_delete( 0 );
	
	for( x = 0 ; x < level.obj_loco.size ; x++)
	{
		if( level.obj_loco[x].completed )
			continue;
		Objective_AdditionalPosition( 0, x, level.obj_loco[x].origin );
		obj_string++;
	}
	
	if( obj_string != 0 )
	{
		objective_add( 0, "active", "" );
		objective_string( 0, &"SUBWAY_OBJECTIVE", obj_string );
		objective_current( 0 );
		
		if( obj_string == 1 )
			flag_set( "boss_sequence_is_last" );
	}
	else
	{
		level.timer destroy();
		wait .5;
		thread credits();
	}
} // update_objective_points()

bomb_triggers()
{
	/*--------------------
	 HANDLE EACH TRIGGER
	----------------------*/
	thread t0();
	thread t1();
	thread t2();
} // bomb_triggers()

t0()
{
	/*--------------------
	 BOMB NEAREST PLAYER
	----------------------*/
	level.obj_loco[ 0 ] waittill ( "trigger" );
	level.player PlaySound( "bomb_defuse" );
	level.obj_loco[ 0 ] trigger_off();
	level.obj_loco[ 0 ].completed = true;
	thread update_objective_points();
} // t0()

t1()
{
	/*--------------------
	 BOMB IN MIDDLE
	----------------------*/
	level.obj_loco[ 1 ] waittill ( "trigger" );
	level.player PlaySound( "bomb_defuse" );
	level.obj_loco[ 1 ] trigger_off();
	level.obj_loco[ 1 ].completed = true;
	thread update_objective_points();
} // t1()

t2()
{
	/*--------------------
	 BOMB FAREST FROM PLAYER
	----------------------*/
	level.obj_loco[ 2 ] waittill ( "trigger" );
	level.player PlaySound( "bomb_defuse" );
	level.obj_loco[ 2 ] trigger_off();
	level.obj_loco[ 2 ].completed = true;
	thread update_objective_points();
} // t2()

lev_timer()
{
	/*--------------------
	 BAESD ON GAMESKILL, CHANGE OBJECTIVE TIME
	----------------------*/
	thread objectives();
	wait 3;
	seconds = undefined;
	switch( level.gameSkill )
	{
		case 0: // Recruit
			seconds = 150;
			break;
		case 1: // Easy
			seconds = 120;
			break;
		case 2: // Hardened
			seconds = 90;
			break;
		case 3: // Veteran
			seconds = 60;
			break;
	}
	assert( isDefined( seconds ) );
	thread timer_director( seconds, &"SUBWAY_TIME_TO_DEFUSE_BOMBS" );
} // lev_timer()

timer_director( seconds, message )
{
	/*--------------------
	 INITIALIZE TIMER
	----------------------*/
	level.timer = maps\_hud_util::get_countdown_hud();
	level.timer.label = message;
	level.timer settenthstimer( seconds );
	level.start_time = gettime();
	
	/*--------------------
	 HANDLE EXPLOISION WHEN OUT OF TIME
	----------------------*/
	wait ( seconds );
	flag_set ( "failed" );
	level.player endon ( "death" );	
	level.timer destroy();
	level.player freezeControls( true );
	level.player setStance( "prone" );
	Earthquake( 0.6, 4, level.player.origin, 512 );
	playFX ( level._effect[ "bomb_explodes" ], level.player.origin );
	level.player playSound( "bomb_explosion" );
	MusicStop( .3 );
	setDvar("ui_deadquote", &"SUBWAY_TIMER_EXPIRED" );
	maps\_utility::missionFailedWrapper();	
} // timer_director()

fan_explosion()
{
	trig = getEnt( "boss_cinematic", "targetname" );
	trig waittill( "trigger" );
	
	/*--------------------
	 DROP THE FAN AND PLAY FX/SOUND
	----------------------*/
	playFX ( level._effect[ "fan_explodes" ], ( 3648, -3704, 128 ) );
	level.player PlaySound( "wall_crumble" );
	Earthquake( 0.6, 4, level.player.origin, 2058 );
	level.fan_b rotateto( level.fan_a.angles, .19 );
	level.fan_b moveto( level.fan_a.origin, .19 );
	level.fan_frame_b rotateto( level.fan_a.angles, .2 );
	level.fan_frame_b moveto( level.fan_a.origin, .2 );
	wait ( .19 );
	level.fan_b hide();
	level.fan_a show();
	level.fan_frame_b hide();
	level.fan_frame_a show();
	
	/*--------------------
	 PLAYER GETS 0WNED BY BLAST
	----------------------*/
	level.player allowCrouch( false );
	level.player allowProne( true );
	level.player allowStand( false );
	level.player allowSprint( false );
	level.player allowjump( false );
	level.player SetStance( "prone" );
	level.player ShellShock( "aftermath", 15 );
	
	thread fade_cinematic();
} // fan_explosion()

fade_cinematic()
{
	/*--------------------
	 FADE IN/OUT AS CINEMATIC HAPPENS
	----------------------*/
	
	wait ( 4 );
	black_overlay = create_overlay_element( "black", 0 );
	black_overlay fadeOverTime(1);
	black_overlay.alpha = 1;
		
	wait ( 2.3 );
	level.player setorigin( (3688, -3414, 43) );
	level.player setplayerangles( (0, 262, 352) );
	guards = getEntArray( "guards", "targetname" );
	for( x = 0 ; x < 2 ; x++ )
	{
		guy = guards[ x ] stalingradspawn();
		spawn_failed( guy );
		guy.ignoreme = true;
		guy.pacifist = true;
		guy setGoalNode( getNode( guy.target, "targetname" ) );
	}
	
	/*
	temp_boss = getEnt( "boss", "targetname" );
	boss = temp_boss stalingradspawn();
	boss.ignoreme = true;
	boss.pacifist = true;
	boss.animName = "human";
	boss SetGoalEntity( level.player );
	thread anim_single_solo( boss, "bossending" );
	*/
	
	level.player switchToWeapon( "Beretta" );
	
	wait ( 0.8 );
	black_overlay destroy();
	black_overlay = create_overlay_element( "black", 0 );
	black_overlay fadeOverTime( 0.5 );
	black_overlay.alpha = 0;
	wait ( 0.5 );
	black_overlay destroy();
	
	if ( !flag( "boss_sequence_is_last" ) )
	{
		// Player gets up slowly
		wait ( 2 );
		level.player allowCrouch( true );
		level.player setStance( "crouch" );
		wait ( 1 );
		level.player setplayerangles( (0, 270, 0) );
		level.player allowStand( true );
		level.player allowSprint( true );
		level.player allowjump( true );
		level.player setStance( "stand" );
		
	}
	else
	{
		// Player must crawl to objective
	}
}

player_loadout()
{
	/*--------------------
	 GIVE THE PLAYER SOME WEAPONS
	----------------------*/
	level.player takeallweapons();
	level.player giveWeapon( "Beretta" );
	level.player giveWeapon( "m4_grenadier" );
	level.player giveWeapon( "fraggrenade" );
	level.player giveWeapon( "flash_grenade" );
	level.player setOffhandSecondaryClass( "flash" );
	level.player switchToWeapon( "m4_grenadier" );
	level.player setViewmodel( "viewmodel_base_viewhands" );
	maps\_load::set_player_viewhand_model( "viewhands_player_usmc" );
} // player_loadout()

train_rumble()
{
	/*--------------------
	 EMULATES A NEARBY TRIAN PASSING BY
	----------------------*/
	while (1)
	{
		rand = RandomIntRange( 20, 40 );
		if( rand == 30 )
		{
			level.player PlaySound( "wall_crumble" );
			wait 0.3;
			Earthquake( 0.3, 4, level.player.origin, 1650 );
		}
		else
			wait 2;
	}
} // train_rumble()

introscreen()
{
	/*--------------------
	 TYPICAL INTROSCREEN
	----------------------*/
	MusicPlay( "intro_song", false );
		
	lines = [];
	lines[ lines.size ] = &"SUBWAY_INTROSCREEN_LINE_1";
	lines[ "date" ] = &"SUBWAY_INTROSCREEN_LINE_2";
	lines[ lines.size ] = &"SUBWAY_INTROSCREEN_LINE_3";
	lines[ lines.size ] = &"SUBWAY_INTROSCREEN_LINE_4";
	level.player PlaySound( "wall_crumble" );
	maps\_introscreen::introscreen_feed_lines( lines );
	
	level.intro_offset = ( 0 );
	fade_time = ( 8 );
	time = ( 8.5 );
	maps\_introscreen::introscreen_generic_fade_in( "black", time, fade_time );
	
	wait ( 0.5 );
	
	flag_set( "introscreen_done" );
	level.player freezeControls( false );
	
	black_overlay = create_overlay_element( "black", 0 );
	black_overlay fadeOverTime( 0.5 );
	black_overlay.alpha = 0;
	wait ( 0.5 );
	black_overlay destroy();
	thread ini_dialogue();
} // introscreen()

credits()
{
	/*--------------------
	 PIGGY-BACKED THE INTROSCREEN, LOOKS BETTER THAN USING PRINT
	----------------------*/
	MusicStop();
	MusicPlayWrapper("credits_song");
	lines = [];
	lines[ lines.size ] = &"SUBWAY_CREDIT1"; // Sparks
	lines[ lines.size ] = &"SUBWAY_CREDIT2"; // AmishThunder
	lines[ lines.size ] = &"SUBWAY_CREDIT3"; // Thanks For Playing
	maps\_introscreen::introscreen_feed_lines( lines );
	
	maps\_introscreen::introscreen_generic_fade_in( "black", 8.5, 8 );
	black_overlay = create_overlay_element( "black", 0 );
	black_overlay fadeOverTime( 0.5 );
	black_overlay.alpha = 1;
	
	/*--------------------
	 QUIT MAP
	----------------------*/
	Changelevel( "" );
} // credits()

create_overlay_element( shader_name, start_alpha )
{
	/*--------------------
	 Jake Keating Tool
	----------------------*/
	overlay = newHudElem();
	overlay.x = 0;
	overlay.y = 0;
	overlay setshader ( shader_name, 640, 480);
	overlay.alignX = "left";
	overlay.alignY = "top";
	overlay.sort = 1;
	overlay.horzAlign = "fullscreen";
	overlay.vertAlign = "fullscreen";
	overlay.alpha = start_alpha;
	overlay.foreground = true;
	return overlay;
} // create_overlay_element()

setTeam( guys, team )
{
	/*--------------------
	 ORIGINALLY INTENDED ON HAVING CIVILIAN AI WALKING AROUND FOR PENALTIES
	 OBSOLETE SCRIPT
	----------------------*/
	if( guys.size == 1 )
		guys.team = team;
	else
		for( x = 0; x < guys.size ; x++ )
		{
			if( isAlive( guys[ x ] ) )
				guys[ x ].team = team;
		}
} // setTeam()

removeWeapon( guys )
{
	/*--------------------
	 ORIGINALLY INTENDED ON HAVING CIVILIAN AI WALKING AROUND FOR PENALTIES
	 OBSOLETE SCRIPT
	----------------------*/
	if( guys.size == 1 )
		guys gun_remove();
	else
		for( x = 0; x < guys.size ; x++ )
		{
			if( isAlive( guys[ x ] ) )
				guys[ x ] gun_remove();
		}
} // RemoveWeapon()

setRandomRun( array )
{
	/*--------------------
	 ORIGINALLY INTENDED ON HAVING CIVILIAN AI WALKING AROUND FOR PENALTIES
	 OBSOLETE SCRIPT
	----------------------*/
	self set_run_anim( array[ RandomInt( array.size ) ], true );
} // setRandomRun